// Admin web UI integration for FastAPI backend.
// Keeps HTML/CSS intact; wires login, auth, data loading and assignments.
(function () {
  const STORAGE_KEYS = {
    flash: "cd_flash_message",
    apiBase: "cd_api_base",
  };

  const state = {
    apiBase: resolveApiBase(),
    auth: loadAuth(),
    club: null,
    coaches: [],
    athletes: [],
    assignments: {}, // athlete_id -> coach_id (session view)
    assigning: new Set(), // prevents duplicate requests
    clubTier: "BASIC",
    benchmark: null,
    accessAudit: [],
  };

  document.addEventListener("DOMContentLoaded", () => {
    showFlashMessageIfAny();
    const page = detectPage();
    if (page === "login") {
      initLoginPage();
    } else if (page === "admin") {
      bindCreateButtons();
      initAdminPage();
    }
  });

  // ---------- Setup / utils ----------
  function resolveApiBase() {
    const params = new URLSearchParams(window.location.search);
    const fromQuery = params.get("api");
    const demoParam = params.get("demo");
    const isElectron =
      typeof navigator !== "undefined" && /electron/i.test(navigator.userAgent || "");
    if (demoParam === "1" || demoParam === "true") {
      const demoBase = isElectron
        ? trimSlash(window.location.origin)
        : trimSlash(`${window.location.origin}/demo`);
      localStorage.setItem(STORAGE_KEYS.apiBase, demoBase);
      return demoBase;
    }
    if (isElectron && window.location.origin && window.location.origin.startsWith("http")) {
      const originBase = trimSlash(window.location.origin);
      localStorage.setItem(STORAGE_KEYS.apiBase, originBase);
      return originBase;
    }
    const host = window.location.hostname || "";
    const isLocal = host === "localhost" || host === "127.0.0.1";
    const allowOverride = isLocal || window.__ALLOW_API_OVERRIDE__ === true;
    if (fromQuery && allowOverride && /^https?:\/\//i.test(fromQuery)) {
      localStorage.setItem(STORAGE_KEYS.apiBase, fromQuery);
      return trimSlash(fromQuery);
    }
    const stored = localStorage.getItem(STORAGE_KEYS.apiBase);
    if (stored && allowOverride && /^https?:\/\//i.test(stored)) {
      if (!isElectron) return trimSlash(stored);
      const origin = window.location.origin || "";
      const allowed = [
        trimSlash(origin),
        trimSlash(`${origin}/demo`),
      ];
      if (allowed.includes(trimSlash(stored))) return trimSlash(stored);
    }
    if (window.__API_BASE__) return trimSlash(window.__API_BASE__);
    if (window.location.origin && window.location.origin.startsWith("http")) {
      const host = window.location.hostname || "";
      const port = window.location.port || "";
      if (!isElectron && (host === "localhost" || host === "127.0.0.1") && port && port !== "8000") {
        return "http://127.0.0.1:8000";
      }
      return trimSlash(window.location.origin);
    }
    return "http://127.0.0.1:8000";
  }

  function trimSlash(base) {
    return String(base || "").replace(/\/+$/, "");
  }

  function buildUrl(path) {
    if (/^https?:\/\//i.test(path)) return path;
    const cleanPath = path.startsWith("/") ? path : `/${path}`;
    return `${state.apiBase}${cleanPath}`;
  }

  function detectPage() {
    const name = (window.location.pathname.split("/").pop() || "").toLowerCase();
    if (name.includes("login")) return "login";
    if (name.includes("admin")) return "admin";
    if (document.querySelector("#login-email") || document.querySelector(".auth-form")) return "login";
    if (document.querySelector(".section-title") && document.querySelector(".section-title").textContent.toLowerCase().includes("dashboard")) return "admin";
    return "other";
  }

  function loadAuth() {
    return {
      role: null,
    };
  }

  function saveAuth({ role }) {
    state.auth = { role: role || null };
  }

  function clearAuth() {
    state.auth = { role: null };
  }

  function flashNext(message) {
    if (!message) return;
    localStorage.setItem(STORAGE_KEYS.flash, message);
  }

  function showFlashMessageIfAny() {
    const msg = localStorage.getItem(STORAGE_KEYS.flash);
    if (msg) {
      showBanner(msg, "error");
      localStorage.removeItem(STORAGE_KEYS.flash);
    }
  }

  async function apiFetch(path, options = {}) {
    const { auth = true, json, headers, ...rest } = options;
    const finalHeaders = { ...(headers || {}) };
    const opts = { ...rest, headers: finalHeaders, credentials: "include" };

    if (json !== undefined) {
      opts.body = JSON.stringify(json);
      opts.method = opts.method || "POST";
      finalHeaders["Content-Type"] = "application/json";
    }
    const resp = await fetch(buildUrl(path), opts);
    const contentType = resp.headers.get("content-type") || "";
    let body = null;
    if (contentType.includes("application/json")) {
      try {
        body = await resp.json();
      } catch (_) {
        body = null;
      }
    } else {
      body = await resp.text();
    }

    if (!resp.ok) {
      if (auth && (resp.status === 401 || resp.status === 403)) {
        logoutAndRedirect("Prihlásenie vypršalo alebo nemáte prístup.");
      }
      const detail = body && body.detail ? body.detail : typeof body === "string" ? body : "Request failed";
      const errObj = new Error(detail || resp.statusText);
      errObj.status = resp.status;
      errObj.body = body;
      throw errObj;
    }
    return body;
  }

  async function fetchWithFallback(paths, options) {
    let lastError = null;
    for (const p of paths) {
      try {
        return await apiFetch(p, options);
      } catch (err) {
        lastError = err;
        if (err && (err.status === 404 || err.status === 405)) {
          continue;
        }
        throw err;
      }
    }
    throw lastError || new Error("Endpoint not available");
  }

  function logoutAndRedirect(message) {
    clearAuth();
    try {
      fetch(buildUrl("/api/logout"), { method: "POST", credentials: "include" });
    } catch (_) {
      // noop
    }
    if (message) flashNext(message);
    if (!window.location.pathname.toLowerCase().includes("login")) {
      window.location.href = "login.html";
    }
  }

  function ensureToastContainer() {
    let c = document.getElementById("cd-toast-container");
    if (!c) {
      c = document.createElement("div");
      c.id = "cd-toast-container";
      c.style.position = "fixed";
      c.style.top = "20px";
      c.style.right = "20px";
      c.style.display = "flex";
      c.style.flexDirection = "column";
      c.style.gap = "8px";
      c.style.zIndex = "9999";
      document.body.appendChild(c);
    }
    return c;
  }

  function showToast(message, type = "info") {
    if (!message) return;
    const container = ensureToastContainer();
    const toast = document.createElement("div");
    toast.className = "cd-toast";
    toast.style.padding = "10px 12px";
    toast.style.borderRadius = "10px";
    toast.style.boxShadow = "0 12px 30px rgba(0,0,0,0.14)";
    toast.style.fontSize = "14px";
    toast.style.color = "#0b1220";
    toast.style.background = "#e8f1ff";
    toast.style.display = "flex";
    toast.style.alignItems = "center";
    toast.style.gap = "8px";
    toast.style.minWidth = "260px";
    toast.style.maxWidth = "360px";
    toast.style.opacity = "0";
    toast.style.transform = "translateY(-6px)";
    toast.style.transition = "opacity 0.2s ease, transform 0.2s ease";

    const icons = { success: "✅", error: "⚠️", warn: "⚠️", info: "ℹ️" };
    const palette = {
      success: { bg: "#e8f5e9", color: "#1b5e20" },
      error: { bg: "#fdecea", color: "#b00020" },
      warn: { bg: "#fff7e6", color: "#8a6d1d" },
      info: { bg: "#e8f1ff", color: "#0b3fa8" },
    };
    const colors = palette[type] || palette.info;

    const icon = document.createElement("span");
    icon.textContent = icons[type] || icons.info;
    toast.appendChild(icon);

    const msg = document.createElement("div");
    msg.textContent = message;
    msg.style.flex = "1";
    toast.appendChild(msg);

    toast.style.background = colors.bg;
    toast.style.color = colors.color;

    container.appendChild(toast);
    requestAnimationFrame(() => {
      toast.style.opacity = "1";
      toast.style.transform = "translateY(0)";
    });
    setTimeout(() => {
      toast.style.opacity = "0";
      toast.style.transform = "translateY(-6px)";
      setTimeout(() => toast.remove(), 220);
    }, 3800);
  }

  function showBanner(message, type = "info") {
    showToast(message, type);
  }

  function friendlyMessage(err, fallback = "Akcia zlyhala.") {
    const status = err?.status;
    if (status === 401 || status === 403) return "You don’t have access to this action.";
    if (status === 404) return "Item not found or not in your club.";
    if (status === 409) return "Already assigned.";
    if (status === 429) return "Too many requests—wait and retry.";
    const msg = err?.message || "";
    if (msg.toLowerCase().includes("network") || msg.toLowerCase().includes("fetch")) {
      return "Network error—check connection.";
    }
    return msg || fallback;
  }

  function lockButton(btn, label = null) {
    if (!btn) return () => {};
    const originalText = btn.textContent;
    btn.disabled = true;
    if (label) btn.textContent = label;
    btn.dataset.loading = "1";
    return () => {
      btn.disabled = false;
      btn.dataset.loading = "0";
      btn.textContent = originalText;
    };
  }

  function openModalShell(title, onClose) {
    const overlay = document.createElement("div");
    overlay.className = "cd-modal";
    const card = document.createElement("div");
    card.className = "cd-modal-card";

    const header = document.createElement("div");
    header.className = "cd-modal-title";
    header.textContent = title;
    card.appendChild(header);

    overlay.appendChild(card);
    document.body.appendChild(overlay);

    const close = () => {
      overlay.remove();
      document.removeEventListener("keydown", onKeyDown);
      if (onClose) onClose();
    };

    const onKeyDown = (e) => {
      if (e.key === "Escape") close();
    };

    overlay.addEventListener("click", (e) => {
      if (e.target === overlay) close();
    });
    document.addEventListener("keydown", onKeyDown);

    return { card, close };
  }

  function openFormModal({ title, fields, submitLabel, onSubmit, onClose }) {
    const { card, close } = openModalShell(title, onClose);
    const form = document.createElement("form");
    form.className = "cd-modal-form";

    const inputs = {};
    fields.forEach((field) => {
      const wrap = document.createElement("div");
      wrap.className = "cd-modal-field";
      const label = document.createElement("label");
      label.textContent = field.label;
      const input = document.createElement("input");
      input.className = "input-control";
      input.type = field.type || "text";
      input.placeholder = field.placeholder || "";
      input.required = Boolean(field.required);
      wrap.append(label, input);
      form.appendChild(wrap);
      inputs[field.name] = input;
    });

    const error = document.createElement("div");
    error.className = "cd-modal-error";
    form.appendChild(error);

    const actions = document.createElement("div");
    actions.className = "cd-modal-actions";
    const cancelBtn = document.createElement("button");
    cancelBtn.type = "button";
    cancelBtn.className = "pill-button pill-ghost";
    cancelBtn.textContent = "Zrušiť";
    cancelBtn.addEventListener("click", close);
    const submitBtn = document.createElement("button");
    submitBtn.type = "submit";
    submitBtn.className = "pill-button";
    submitBtn.textContent = submitLabel || "Uložiť";
    actions.append(cancelBtn, submitBtn);
    form.appendChild(actions);

    form.addEventListener("submit", async (e) => {
      e.preventDefault();
      error.textContent = "";
      const values = {};
      for (const field of fields) {
        const value = (inputs[field.name]?.value || "").trim();
        if (field.required && !value) {
          error.textContent = "Vyplň povinné polia.";
          inputs[field.name]?.focus();
          return;
        }
        values[field.name] = value;
      }

      const release = lockButton(submitBtn, "Ukladám…");
      try {
        await onSubmit(values);
        close();
      } catch (err) {
        error.textContent = friendlyMessage(err, "Akcia zlyhala.");
      } finally {
        release();
      }
    });

    card.appendChild(form);
    const firstInput = Object.values(inputs)[0];
    if (firstInput) firstInput.focus();
  }

  function openCredentialsModal(title, creds) {
    if (!creds || (!creds.username && !creds.password)) return;
    const { card, close } = openModalShell(title);
    const text = document.createElement("textarea");
    text.className = "input-control cd-modal-textarea";
    text.readOnly = true;
    text.rows = 3;
    text.value = `Prihlasovacie meno: ${creds.username || ""}\nHeslo: ${creds.password || ""}`;
    card.appendChild(text);

    const actions = document.createElement("div");
    actions.className = "cd-modal-actions";
    const copyBtn = document.createElement("button");
    copyBtn.type = "button";
    copyBtn.className = "pill-button";
    copyBtn.textContent = "Skopírovať";
    copyBtn.addEventListener("click", async () => {
      try {
        await navigator.clipboard.writeText(text.value);
        showToast("Skopírované do schránky.", "success");
      } catch (_) {
        showToast("Kopírovanie zlyhalo.", "error");
      }
    });
    const closeBtn = document.createElement("button");
    closeBtn.type = "button";
    closeBtn.className = "pill-button pill-ghost";
    closeBtn.textContent = "Zavrieť";
    closeBtn.addEventListener("click", close);
    actions.append(copyBtn, closeBtn);
    card.appendChild(actions);
  }

  function formatName(entity, fallback = "—") {
    if (!entity) return fallback;
    const first = (entity.first_name || "").trim();
    const last = (entity.last_name || "").trim();
    const full = `${first} ${last}`.trim();
    if (full) return full;
    return entity.name || fallback;
  }

  function formatUsername(entity) {
    if (!entity) return "";
    return entity.username || entity.user_id || entity.id || "";
  }

  function shortName(entity) {
    const f = (entity.first_name || "").trim();
    const l = (entity.last_name || "").trim();
    if (f && l) return `${f[0]}. ${l}`;
    if (l) return l;
    return f || formatUsername(entity) || "—";
  }

  function formatRoleLabel(role) {
    const value = String(role || "").toUpperCase();
    if (value === "ADMIN") return "Admin";
    if (value === "COACH") return "Tréner";
    if (value === "ATHLETE") return "Športovec";
    return value || "—";
  }

  function formatLastLogin(value) {
    if (!value) return "—";
    const dt = new Date(value);
    if (Number.isNaN(dt.getTime())) return value;
    return dt.toLocaleString("sk-SK");
  }

  function sparkline(values, width = 140, height = 40, color = "#06aed5") {
    if (!values || !values.length) return "";
    const max = Math.max(...values);
    const min = Math.min(...values);
    const range = max === min ? 1 : max - min;
    const stepX = values.length > 1 ? width / (values.length - 1) : width;
    const pts = values
      .map((v, i) => {
        const x = i * stepX;
        const y = height - ((v - min) / range) * height;
        return `${x.toFixed(1)},${y.toFixed(1)}`;
      })
      .join(" ");
    return `<svg viewBox="0 0 ${width} ${height}" class="sparkline" preserveAspectRatio="none"><polyline fill="none" stroke="${color}" stroke-width="2" points="${pts}" /></svg>`;
  }

  function findSectionCard(kickerText) {
    return Array.from(document.querySelectorAll(".section-card")).find((card) => {
      const kicker = card.querySelector(".section-kicker");
      return kicker && kicker.textContent.toLowerCase().includes(kickerText.toLowerCase());
    });
  }

  function createEmptyRow(table, cols, message) {
    const row = document.createElement("div");
    row.className = "table-grid table-row";
    row.style.setProperty("--cols", cols);
    const cell = document.createElement("div");
    cell.textContent = message;
    cell.style.gridColumn = "1 / -1";
    row.appendChild(cell);
    table.appendChild(row);
  }

  // ---------- Login ----------
  function initLoginPage() {
    const emailInput = document.getElementById("login-email");
    const passwordInput = document.getElementById("login-password");
    const form = emailInput ? emailInput.closest("form") : null;
    const submitBtn = form ? form.querySelector(".btn-primary") : null;
    if (!form || !submitBtn) return;

    // Already logged in as admin? verify session then redirect.
    (async () => {
      try {
        const res = await fetch(buildUrl("/api/me"), { credentials: "include" });
        if (!res.ok) return;
        const me = await res.json();
        if ((me.role || "").toUpperCase() === "ADMIN") {
          showBanner("Ste prihlásený ako admin. Presmerovávam…", "info");
          setTimeout(() => (window.location.href = "admin.html"), 400);
        }
      } catch (_) {
        // ignore
      }
    })();

    const feedback = document.createElement("div");
    feedback.style.marginTop = "8px";
    feedback.style.fontSize = "14px";
    feedback.style.minHeight = "20px";
    form.appendChild(feedback);

    const setFeedback = (msg, type = "info") => {
      feedback.textContent = msg;
      feedback.style.color = type === "error" ? "#b00020" : "#0b3fa8";
    };

    const params = new URLSearchParams(window.location.search);
    const isDemo = ["1", "true"].includes(String(params.get("demo") || "").toLowerCase());
    const autoLogin = ["1", "true"].includes(String(params.get("autologin") || "").toLowerCase());
    const autoLoginKey = "cd_demo_autologin_done";
    if (isDemo && autoLogin && !sessionStorage.getItem(autoLoginKey)) {
      sessionStorage.setItem(autoLoginKey, "1");
      setFeedback("Demo sa načítava...", "info");
      fetchWithFallback(
        ["/login", "/api/login"],
        {
          auth: false,
          json: { username: "demo_admin", password: "DemoAdmin2025!" },
        }
      )
        .then((data) => {
          const role = (data.role || "").toUpperCase();
          if (role !== "ADMIN") {
            throw new Error("Demo login failed");
          }
          saveAuth({ role });
          setFeedback("Demo pripravené, presmerovávam…", "info");
          setTimeout(() => (window.location.href = "admin.html"), 400);
        })
        .catch((err) => {
          console.warn("[CoachDesk] Demo auto-login failed:", err);
          setFeedback("Demo sa nepodarilo prihlásiť. Skús prihlásenie manuálne.", "error");
        });
    }

    async function handleLogin(e) {
      e.preventDefault();
      const username = (emailInput.value || "").trim();
      const password = (passwordInput.value || "").trim();
      if (!username || !password) {
        setFeedback("Zadaj prihlasovacie meno alebo e-mail a heslo.", "error");
        return;
      }
      submitBtn.disabled = true;
      const originalLabel = submitBtn.textContent;
      submitBtn.textContent = "Prihlasujem…";
      setFeedback("");
      try {
        const data = await fetchWithFallback(
          ["/login", "/api/login"],
          {
            auth: false,
            json: { username, password },
          }
        );
        const role = (data.role || "").toUpperCase();
        if (role !== "ADMIN") {
          clearAuth();
          setFeedback("Prístup len pre adminov. Použi admin konto.", "error");
          return;
        }
        saveAuth({ role });
        console.info("[CoachDesk] Admin login OK; session cookie set.");
        setFeedback("Prihlásenie úspešné, presmerovávam…", "info");
        setTimeout(() => (window.location.href = "admin.html"), 400);
      } catch (err) {
        setFeedback(err.message || "Prihlásenie zlyhalo.", "error");
      } finally {
        submitBtn.disabled = false;
        submitBtn.textContent = originalLabel;
      }
    }

    submitBtn.addEventListener("click", handleLogin);
    form.addEventListener("submit", handleLogin);

    const regForm = document.querySelector(".auth-card--ghost .auth-form");
    const regBtn = regForm ? regForm.querySelector(".btn-outline") : null;
    if (regForm && regBtn) {
      const regFeedback = document.createElement("div");
      regFeedback.style.marginTop = "8px";
      regFeedback.style.fontSize = "14px";
      regFeedback.style.minHeight = "20px";
      regForm.appendChild(regFeedback);

      const setRegFeedback = (msg, type = "info") => {
        regFeedback.textContent = msg || "";
        if (type === "error") regFeedback.style.color = "#b00020";
        else if (type === "success") regFeedback.style.color = "#1b5e20";
        else regFeedback.style.color = "#0b3fa8";
      };

      const getVal = (id) => {
        const el = document.getElementById(id);
        return el ? el.value.trim() : "";
      };

      async function handleRegistration(e) {
        e.preventDefault();
        const clubName = getVal("club-name");
        const contactName = getVal("club-contact");
        const contactRole = getVal("club-role");
        const email = getVal("club-email");
        const sizeRaw = getVal("club-size");
        const swimmersCount = sizeRaw ? Number(sizeRaw) : undefined;

        if (!clubName || !email) {
          setRegFeedback("Vyplňte názov klubu a e-mail.", "error");
          return;
        }

        const payload = {
          club_name: clubName,
          contact_name: contactName,
          contact_role: contactRole,
          email,
        };
        if (Number.isFinite(swimmersCount)) {
          payload.swimmers_count = swimmersCount;
        }
        if (email) {
          payload.admin_username = email;
        }

        regBtn.disabled = true;
        const original = regBtn.textContent;
        regBtn.textContent = "Odosielam…";
        setRegFeedback("Odosielam žiadosť o registráciu...", "info");
        let data = null;
        let lastError = null;
        for (const path of ["/register/club", "/api/register/club"]) {
          try {
            const res = await fetch(buildUrl(path), {
              method: "POST",
              headers: { "Content-Type": "application/json" },
              body: JSON.stringify(payload),
              credentials: "include",
            });
            const body = await res.json().catch(() => ({}));
            if (!res.ok) {
              throw new Error(body.detail || res.statusText || "Registrácia zlyhala");
            }
            data = body;
            break;
          } catch (err) {
            lastError = err;
          }
        }
        if (!data) {
          setRegFeedback(lastError?.message || "Registrácia zlyhala.", "error");
          regBtn.disabled = false;
          regBtn.textContent = original;
          return;
        }

        localStorage.setItem(STORAGE_KEYS.apiBase, state.apiBase);
        const role = (data.role || "ADMIN").toUpperCase();
        saveAuth({ role });
        const creds = data.admin_credentials || {};
        const credMsg =
          creds.username && creds.password ? ` Prihlasovacie údaje: ${creds.username} / ${creds.password}` : "";
        setRegFeedback(`Klub vytvorený. Presmerovávam do admin zóny.${credMsg}`, "success");
        regBtn.textContent = original;
        regBtn.disabled = false;
        setTimeout(() => (window.location.href = "admin.html"), 600);
      }

      regBtn.addEventListener("click", handleRegistration);
      regForm.addEventListener("submit", handleRegistration);
    }
  }

  async function validateAdminRole() {
    const me = await fetchWithFallback(["/api/me", "/me"], { auth: true });
    const role = (me.role || "").toUpperCase();
    if (role !== "ADMIN") {
      throw new Error("Forbidden");
    }
    return me;
  }

  // ---------- Admin ----------
  async function initAdminPage() {
    try {
      const me = await validateAdminRole();
      state.auth.role = me.role;
      if (me.club_tier && me.club_tier.tier) state.clubTier = (me.club_tier.tier || "BASIC").toUpperCase();
      console.info("[CoachDesk] Overená admin rola pre", me.username || me.user_id);
    } catch (err) {
      logoutAndRedirect("Prístup len pre adminov.");
      return;
    }

    try {
      const [club, coaches, athletes] = await Promise.all([
        fetchClubSafe(),
        loadCoaches(),
        loadAthletes(),
      ]);
      state.club = club;
      state.coaches = coaches || [];
      state.athletes = athletes || [];
      console.info("[CoachDesk] Načítaní tréneri:", state.coaches.length, " | športovci:", state.athletes.length);
      renderCoaches();
      renderAthletes();
      renderAssignments();
      renderExportSelectors();
      renderStats();
      await loadBenchmark();
      await loadAccessAudit();
      bindExportButtons();
      bindCreateButtons();
    } catch (err) {
      showBanner(err.message || "Nepodarilo sa načítať dáta.", "error");
    }
  }

  async function fetchClubSafe() {
    try {
      return await fetchWithFallback(["/api/club", "/club"], { auth: true });
    } catch (err) {
      return null;
    }
  }

  async function loadCoaches() {
    const data = await fetchWithFallback(
      ["/admin/coaches", "/api/club/coaches", "/club/coaches"],
      { auth: true }
    );
    return Array.isArray(data) ? data : [];
  }

  async function loadAthletes() {
    const data = await fetchWithFallback(
      ["/admin/athletes", "/api/club/athletes", "/club/athletes"],
      { auth: true }
    );
    return Array.isArray(data) ? data : [];
  }

  function renderCoaches() {
    const card = findSectionCard("Správa trénerov");
    if (!card) return;
    const table = card.querySelector(".table");
    if (!table) return;
    table.querySelectorAll(".table-row").forEach((row) => row.remove());
    const cols = "1.2fr 1fr 0.8fr 0.9fr 1fr";
    if (!state.coaches.length) {
      createEmptyRow(table, cols, "No coaches yet—add coach to start assignments.");
      return;
    }
    state.coaches.forEach((coach) => {
      const row = document.createElement("div");
      row.className = "table-grid table-row";
      row.style.setProperty("--cols", cols);

      const nameCell = document.createElement("div");
      const nameStrong = document.createElement("strong");
      nameStrong.textContent = formatName(coach, "Tréner");
      const nameHandle = document.createElement("span");
      nameHandle.className = "muted";
      nameHandle.textContent = `@${formatUsername(coach) || "user"}`;
      nameCell.append(nameStrong, document.createElement("br"), nameHandle);

      const clubCell = document.createElement("div");
      clubCell.textContent = (state.club && state.club.name) || "Môj klub";

      const statusCell = document.createElement("div");
      const badge = document.createElement("span");
      badge.className = "status-badge is-active";
      badge.textContent = "Aktívny";
      statusCell.appendChild(badge);

      const assignedCell = document.createElement("div");
      const assignedCount = Object.values(state.assignments || {}).filter((cid) => cid === coach.id).length;
      assignedCell.textContent = assignedCount ? `${assignedCount}` : "—";

      const actions = document.createElement("div");
      actions.className = "table-actions";
      const editBtn = document.createElement("button");
      editBtn.className = "pill-button pill-ghost";
      editBtn.textContent = "✏️ Upraviť";
      editBtn.disabled = true;
      const deactivateBtn = document.createElement("button");
      deactivateBtn.className = "pill-button pill-danger";
      deactivateBtn.textContent = "⛔ Deaktivovať";
      deactivateBtn.disabled = true;
      actions.append(editBtn, deactivateBtn);

      row.append(nameCell, clubCell, statusCell, assignedCell, actions);
      table.appendChild(row);
    });
  }

  function renderAthletes() {
    const card = findSectionCard("Správa športovcov");
    if (!card) return;
    const table = card.querySelector(".table");
    if (!table) return;
    table.querySelectorAll(".table-row").forEach((row) => row.remove());
    const cols = "1.2fr 0.7fr 0.8fr 0.9fr 0.9fr 1fr";

    if (!state.athletes.length) {
      createEmptyRow(table, cols, "No athletes yet—add your first athlete to assign coaches.");
      return;
    }

    state.athletes.forEach((athlete) => {
      const row = document.createElement("div");
      row.className = "table-grid table-row";
      row.style.setProperty("--cols", cols);

      const nameCell = document.createElement("div");
      const nameStrong = document.createElement("strong");
      nameStrong.textContent = formatName(athlete, "Športovec");
      const nameHandle = document.createElement("span");
      nameHandle.className = "muted";
      nameHandle.textContent = `@${formatUsername(athlete) || athlete.id}`;
      nameCell.append(nameStrong, document.createElement("br"), nameHandle);

      const yearCell = document.createElement("div");
      yearCell.textContent = athlete.birth_date ? athlete.birth_date.split("-")[0] : athlete.category || "—";

      const disciplineCell = document.createElement("div");
      disciplineCell.textContent = athlete.dominant_style || "—";

      const clubCell = document.createElement("div");
      clubCell.textContent = (state.club && state.club.name) || "Môj klub";

      const statusCell = document.createElement("div");
      const badge = document.createElement("span");
      badge.className = "status-badge is-active";
      badge.textContent = "Aktívny";
      statusCell.appendChild(badge);

      const coachCell = document.createElement("div");
      const assignedCoachId = state.assignments[athlete.id];
      const coach = state.coaches.find((c) => c.id === assignedCoachId);
      const coachLabel = document.createElement("span");
      coachLabel.className = "tag tag-soft";
      coachLabel.textContent = coach ? shortName(coach) : "Nepriradený";
      coachCell.className = "table-actions";
      coachCell.appendChild(coachLabel);

      row.append(nameCell, yearCell, disciplineCell, clubCell, statusCell, coachCell);
      table.appendChild(row);
    });
  }

  function renderAssignments() {
    const card = findSectionCard("Priradenie športovec");
    if (!card) return;
    const table = card.querySelector(".table");
    if (!table) return;
    table.querySelectorAll(".table-row").forEach((row) => row.remove());
    const cols = "1.2fr 1fr 1fr 1fr";

    if (!state.athletes.length) {
      createEmptyRow(table, cols, "No athletes yet—add your first athlete to assign coaches.");
      return;
    }
    if (!state.coaches.length) {
      createEmptyRow(table, cols, "No coaches yet—add coach to start assignments.");
      return;
    }

    state.athletes.forEach((athlete) => {
      const row = document.createElement("div");
      row.className = "table-grid table-row";
      row.style.setProperty("--cols", cols);

      const nameCell = document.createElement("div");
      const nameStrong = document.createElement("strong");
      nameStrong.textContent = formatName(athlete, "Športovec");
      const nameHandle = document.createElement("span");
      nameHandle.className = "muted";
      nameHandle.textContent = `@${formatUsername(athlete) || athlete.id}`;
      nameCell.append(nameStrong, document.createElement("br"), nameHandle);

      const clubCell = document.createElement("div");
      clubCell.textContent = (state.club && state.club.name) || "Môj klub";

      const statusCell = document.createElement("div");
      const badge = document.createElement("span");
      const assignedCoachId = state.assignments[athlete.id];
      const assignedCoach = state.coaches.find((c) => c.id === assignedCoachId);
      if (assignedCoach) {
        badge.className = "status-badge is-assigned";
        badge.textContent = `Priradený: ${shortName(assignedCoach)}`;
      } else {
        badge.className = "status-badge is-unassigned";
        badge.textContent = "Nepriradený";
      }
      statusCell.appendChild(badge);

      const selectCell = document.createElement("div");
      const select = document.createElement("select");
      select.className = "table-select";
      const placeholder = document.createElement("option");
      placeholder.textContent = assignedCoach ? "Zmeniť trénera" : "Vybrať trénera";
      placeholder.disabled = true;
      placeholder.selected = !assignedCoach;
      select.appendChild(placeholder);
      state.coaches.forEach((coach) => {
        const opt = document.createElement("option");
        opt.value = coach.id;
        opt.textContent = shortName(coach);
        if (assignedCoachId === coach.id) opt.selected = true;
        select.appendChild(opt);
      });
      select.addEventListener("change", () => {
        const coachId = select.value;
        if (!coachId || state.assigning.has(athlete.id)) return;
        assignAthlete(athlete, coachId, select);
      });
      selectCell.appendChild(select);

      row.append(nameCell, clubCell, statusCell, selectCell);
      table.appendChild(row);
    });
  }

  function renderStats() {
    const statCoaches = document.getElementById("stat-coaches");
    const statAthletes = document.getElementById("stat-athletes");
    const statTier = document.getElementById("stat-tier");
    const statBench = document.getElementById("stat-benchmark");
    if (statCoaches) statCoaches.textContent = state.coaches.length || "0";
    if (statAthletes) statAthletes.textContent = state.athletes.length || "0";
    if (statTier) statTier.textContent = state.clubTier || "BASIC";
    if (statBench) statBench.textContent = state.benchmark?.metrics?.avg_weekly_load ?? "–";
  }

  function renderAccessAudit() {
    const body = document.getElementById("access-table-body");
    if (!body) return;
    body.innerHTML = "";
    const cols = "1.1fr 0.9fr 0.9fr 1fr 1fr";
    if (!state.accessAudit.length) {
      createEmptyRow(body, cols, "Žiadne záznamy prístupov.");
      return;
    }
    state.accessAudit.forEach((entry) => {
      const row = document.createElement("div");
      row.className = "table-grid table-row";
      row.style.setProperty("--cols", cols);

      const userCell = document.createElement("div");
      const displayName = entry.name || entry.username || "—";
      const userStrong = document.createElement("strong");
      userStrong.textContent = displayName;
      userCell.appendChild(userStrong);
      if (entry.username && entry.username !== displayName) {
        const br = document.createElement("br");
        const handle = document.createElement("span");
        handle.className = "muted";
        handle.textContent = `@${entry.username}`;
        userCell.append(br, handle);
      }

      const roleCell = document.createElement("div");
      const roleTag = document.createElement("span");
      roleTag.className = "tag tag-soft";
      roleTag.textContent = formatRoleLabel(entry.role);
      roleCell.appendChild(roleTag);

      const clubCell = document.createElement("div");
      clubCell.textContent = entry.club || "—";

      const accessCell = document.createElement("div");
      accessCell.textContent = entry.access || "—";

      const loginCell = document.createElement("div");
      loginCell.textContent = formatLastLogin(entry.last_login);

      row.append(userCell, roleCell, clubCell, accessCell, loginCell);
      body.appendChild(row);
    });
  }

  async function loadAccessAudit() {
    try {
      const data = await apiFetch("/api/club/access", { auth: true });
      state.accessAudit = Array.isArray(data) ? data : [];
      renderAccessAudit();
    } catch (err) {
      state.accessAudit = [];
      renderAccessAudit();
      showToast(friendlyMessage(err, "Nepodarilo sa načítať prístupy."), "error");
    }
  }

  function renderExportSelectors() {
    const athSelect = document.getElementById("select-athlete-export");
    const coachSelect = document.getElementById("select-coach-export");
    if (athSelect) {
      athSelect.textContent = "";
      const placeholder = document.createElement("option");
      placeholder.value = "";
      placeholder.textContent = "Vyber športovca";
      athSelect.appendChild(placeholder);
      state.athletes.forEach((athlete) => {
        const option = document.createElement("option");
        option.value = athlete.id;
        option.textContent = formatName(athlete, athlete.id);
        athSelect.appendChild(option);
      });
    }
    if (coachSelect) {
      coachSelect.textContent = "";
      const placeholder = document.createElement("option");
      placeholder.value = "";
      placeholder.textContent = "Vyber trénera";
      coachSelect.appendChild(placeholder);
      state.coaches.forEach((coach) => {
        const option = document.createElement("option");
        option.value = coach.id;
        option.textContent = formatName(coach, coach.id);
        coachSelect.appendChild(option);
      });
    }
  }

  async function loadBenchmark() {
    const summary = document.getElementById("benchmark-summary");
    const chart = document.getElementById("benchmark-chart");
    if (!state.club || !state.club.id) {
      if (summary) summary.textContent = "Chýba klubový profil.";
      return;
    }
    try {
      const data = await apiFetch(`/admin/club/${state.club.id}/benchmark`, { auth: true });
      state.benchmark = data;
      renderBenchmark();
      renderStats();
      if (summary) summary.textContent = "Benchmark aktualizovaný.";
      showToast("Benchmark loaded.", "success");
      if (chart && data?.weeks) {
        const vals = Object.values(data.weeks);
        chart.innerHTML = sparkline(vals, 220, 60, "#06aed5");
      }
    } catch (err) {
      const msg = friendlyMessage(err, "Benchmark nedostupný.");
      if (summary) summary.textContent = msg;
      showToast(msg, "error");
    }
  }

  function renderBenchmark() {
    const box = document.getElementById("benchmark-summary");
    if (!box) return;
    const m = state.benchmark?.metrics;
    if (!m) {
      box.textContent = "Benchmark načítam po prihlásení.";
      return;
    }
    box.textContent = "";
    const rows = [
      ["Priem. týždenný objem", m.avg_weekly_load],
      ["Injury proxy", m.injury_risk_proxy],
      ["Performance proxy", m.performance_proxy],
    ];
    rows.forEach(([label, value]) => {
      const row = document.createElement("div");
      row.className = "muted";
      const strong = document.createElement("strong");
      strong.textContent = value ?? "—";
      row.textContent = `${label}: `;
      row.appendChild(strong);
      box.appendChild(row);
    });
  }

  async function downloadExport(scope, fmt, triggerBtn) {
    const release = lockButton(triggerBtn, "Preparing…");
    try {
      showToast("Preparing export…", "info");
      let path = "";
      if (scope === "athlete") {
        const athSelect = document.getElementById("select-athlete-export");
        const val = athSelect ? athSelect.value : "";
        if (!val) throw new Error("Vyberte športovca.");
        path = `/admin/export/athlete/${val}${fmt === "pdf" ? "/pdf" : ""}`;
      } else if (scope === "coach") {
        const coachSelect = document.getElementById("select-coach-export");
        const val = coachSelect ? coachSelect.value : "";
        if (!val) throw new Error("Vyberte trénera.");
        path = `/admin/export/coach/${val}${fmt === "pdf" ? "/pdf" : ""}`;
      } else {
        const cid = state.club?.id;
        if (!cid) throw new Error("Chýba klubové ID.");
        path = `/admin/export/club/${cid}${fmt === "pdf" ? "/pdf" : ""}`;
      }
      const res = await fetch(buildUrl(path), { credentials: "include" });
      if (!res.ok) throw new Error(`HTTP ${res.status}`);
      const blob = await res.blob();
      const a = document.createElement("a");
      a.href = URL.createObjectURL(blob);
      a.download = `${scope}-export.${fmt}`;
      document.body.appendChild(a);
      a.click();
      a.remove();
      showToast("Export ready.", "success");
    } catch (err) {
      showToast(friendlyMessage(err, "Export zlyhal."), "error");
    }
    release();
  }

  function bindExportButtons() {
    const athCsv = document.getElementById("btn-export-athlete-csv");
    const athPdf = document.getElementById("btn-export-athlete-pdf");
    const coachCsv = document.getElementById("btn-export-coach-csv");
    const coachPdf = document.getElementById("btn-export-coach-pdf");
    const clubCsv = document.getElementById("btn-export-club-csv");
    const clubPdf = document.getElementById("btn-export-club-pdf");
    if (athCsv) athCsv.onclick = (e) => downloadExport("athlete", "csv", e.currentTarget);
    if (athPdf) athPdf.onclick = (e) => downloadExport("athlete", "pdf", e.currentTarget);
    if (coachCsv) coachCsv.onclick = (e) => downloadExport("coach", "csv", e.currentTarget);
    if (coachPdf) coachPdf.onclick = (e) => downloadExport("coach", "pdf", e.currentTarget);
    if (clubCsv) clubCsv.onclick = (e) => downloadExport("club", "csv", e.currentTarget);
    if (clubPdf) clubPdf.onclick = (e) => downloadExport("club", "pdf", e.currentTarget);
  }

  function bindCreateButtons() {
    const addCoach = document.getElementById("btn-add-coach");
    const addAthlete = document.getElementById("btn-add-athlete");
    if (addCoach && !addCoach.dataset.bound) {
      addCoach.dataset.bound = "1";
      addCoach.onclick = () => createCoach(addCoach);
    }
    if (addAthlete && !addAthlete.dataset.bound) {
      addAthlete.dataset.bound = "1";
      addAthlete.onclick = () => createAthlete(addAthlete);
    }
  }

  async function createCoach(triggerBtn) {
    const release = lockButton(triggerBtn);
    openFormModal({
      title: "Nový tréner",
      submitLabel: "Vytvoriť trénera",
      onClose: release,
      fields: [
        { name: "first_name", label: "Meno", placeholder: "Jana", required: true },
        { name: "last_name", label: "Priezvisko", placeholder: "Nováková", required: false },
        { name: "email", label: "E-mail", placeholder: "trener@klub.sk", required: false },
        { name: "phone", label: "Telefón", placeholder: "+421...", required: false },
      ],
      onSubmit: async (values) => {
        const data = await apiFetch("/api/club/coaches", {
          auth: true,
          json: {
            first_name: values.first_name,
            last_name: values.last_name || "",
            email: values.email || "",
            phone: values.phone || "",
          },
        });
        showToast("Tréner vytvorený.", "success");
        openCredentialsModal("Nový tréner", data?.credentials);
        state.coaches = await loadCoaches();
        renderCoaches();
        renderAssignments();
        await loadAccessAudit();
        renderExportSelectors();
        renderStats();
      },
    });
  }

  async function createAthlete(triggerBtn) {
    const release = lockButton(triggerBtn);
    openFormModal({
      title: "Nový športovec",
      submitLabel: "Vytvoriť športovca",
      onClose: release,
      fields: [
        { name: "first_name", label: "Meno", placeholder: "Laura", required: true },
        { name: "last_name", label: "Priezvisko", placeholder: "Kováčová", required: true },
        { name: "gender", label: "Pohlavie", placeholder: "M/Ž", required: false },
        { name: "category", label: "Kategória", placeholder: "U18", required: false },
        { name: "dominant_style", label: "Dominantná disciplína", placeholder: "200m kraul", required: false },
        {
          name: "swimrankings_profile_url",
          label: "Swimrankings profil URL",
          placeholder: "https://www.swimrankings.net/index.php?page=athleteDetail&athleteId=12345",
          required: false,
          type: "url",
        },
      ],
      onSubmit: async (values) => {
        const data = await apiFetch("/api/club/athletes", {
          auth: true,
          json: {
            first_name: values.first_name,
            last_name: values.last_name,
            gender: values.gender || "",
            category: values.category || "",
            dominant_style: values.dominant_style || "",
            swimrankings_profile_url: values.swimrankings_profile_url || "",
          },
        });
        showToast("Športovec vytvorený.", "success");
        openCredentialsModal("Nový športovec", data?.credentials);
        state.athletes = await loadAthletes();
        renderAthletes();
        renderAssignments();
        await loadAccessAudit();
        renderExportSelectors();
        renderStats();
      },
    });
  }

  async function assignAthlete(athlete, coachId, selectEl) {
    state.assigning.add(athlete.id);
    selectEl.disabled = true;
    showToast("Assigning athlete…", "info");
    try {
      await apiFetch("/admin/assign-athlete", {
        auth: true,
        json: { coach_id: coachId, athlete_id: athlete.id },
      });
      state.assignments[athlete.id] = coachId;
      const coach = state.coaches.find((c) => c.id === coachId);
      showToast(`Assigned: ${formatName(athlete)} → ${shortName(coach || {})}`, "success");
      console.info("[CoachDesk] Priradenie OK:", athlete.id, "->", coachId);
      renderAssignments();
      renderCoaches();
      renderAthletes();
    } catch (err) {
      if (err && err.status === 409) {
        showToast("Already assigned.", "info");
      } else {
        showToast(friendlyMessage(err, "Priradenie zlyhalo."), "error");
      }
    } finally {
      state.assigning.delete(athlete.id);
      selectEl.disabled = false;
    }
  }
})();
