document.addEventListener("DOMContentLoaded", () => {
  document.querySelectorAll(".live-headline").forEach((el) => {
    el.style.transition = "opacity 0.8s ease, transform 0.8s ease";
    requestAnimationFrame(() => {
      el.style.opacity = "1";
      el.style.transform = "translateY(-4px)";
    });
  });

  const weeklyValues = [60, 65, 72, 85, 78, 70, 55];
  const chart = document.getElementById("weekly-load-chart");
  const pathEl = document.getElementById("weekly-path");
  const width = 140;
  const height = 80;
  const padding = 12;

  if (chart && pathEl) {
    const min = Math.min(...weeklyValues);
    const max = Math.max(...weeklyValues);
    const range = Math.max(max - min, 1);
    const points = weeklyValues.map((v, i) => {
      const x = (width / (weeklyValues.length - 1)) * i;
      const y = height - padding - ((v - min) / range) * (height - padding * 2);
      return { x, y, v };
    });

    const buildSmoothPath = (pts) => {
      if (!pts.length) return "";
      let d = `M ${pts[0].x} ${pts[0].y}`;
      for (let i = 1; i < pts.length; i++) {
        const prev = pts[i - 1];
        const curr = pts[i];
        const midX = (prev.x + curr.x) / 2;
        const midY = (prev.y + curr.y) / 2;
        d += ` Q ${prev.x} ${prev.y} ${midX} ${midY}`;
      }
      d += ` T ${pts[pts.length - 1].x} ${pts[pts.length - 1].y}`;
      return d;
    };

    const lineD = buildSmoothPath(points);
    pathEl.setAttribute("d", lineD);

    const len = pathEl.getTotalLength();
    pathEl.style.strokeDasharray = `${len}`;
    pathEl.style.strokeDashoffset = len;
    requestAnimationFrame(() => {
      pathEl.style.strokeDashoffset = "0";
    });
  }

  const slides = Array.from(document.querySelectorAll(".beta-slide"));
  const dots = Array.from(document.querySelectorAll(".beta-dot"));
  let slideIndex = 0;

  const showSlide = (idx) => {
    slideIndex = (idx + slides.length) % slides.length;
    slides.forEach((s, i) => s.classList.toggle("is-active", i === slideIndex));
    dots.forEach((d, i) => d.classList.toggle("is-active", i === slideIndex));
  };

  dots.forEach((dot, i) => {
    dot.addEventListener("click", () => showSlide(i));
  });

  setInterval(() => showSlide(slideIndex + 1), 4800);

  const submitButtons = [
    document.getElementById("beta-submit-primary"),
    document.getElementById("beta-submit-secondary"),
  ];
  const statusEl = document.getElementById("beta-status");
  const getVal = (id, max = 1000) => {
    const el = document.getElementById(id);
    return (el && el.value ? el.value : "").toString().trim().slice(0, max);
  };
  const setStatus = (text, isError = false) => {
    if (!statusEl) return;
    statusEl.textContent = text || "";
    statusEl.style.display = text ? "block" : "none";
    statusEl.style.color = isError ? "#b91c1c" : "#0f5132";
  };
  let isSubmitting = false;
  const toggleButtons = (disabled) => {
    submitButtons.forEach((btn) => {
      if (btn) {
        btn.disabled = disabled;
      }
    });
  };
  const FORMSPREE_ENDPOINT = "https://formspree.io/f/mlgrnvbg";
  const EMAIL_PATTERN = /^[^@\s]+@[^@\s]+\.[^@\s]+$/;
  const handleSubmit = async () => {
    if (isSubmitting) return;
    const payload = {
      club: getVal("beta-club"),
      contact: getVal("beta-contact"),
      email: getVal("beta-email"),
      club_size: getVal("beta-size"),
      start_date: getVal("beta-start"),
      usage_intent: getVal("beta-use"),
      current_planning: getVal("beta-process"),
      needs: getVal("beta-needs"),
      ai_attitude: getVal("beta-ai"),
    };
    if (!payload.club || !payload.contact || !payload.email) {
      setStatus("Vyplňte prosím klub, kontaktnú osobu a e-mail.", true);
      return;
    }
    if (!EMAIL_PATTERN.test(payload.email)) {
      setStatus("Zadajte prosím platný e-mail.", true);
      return;
    }
    isSubmitting = true;
    toggleButtons(true);
    setStatus("Odosielame vaše odpovede...", false);
    try {
      const notes = [
        payload.club_size ? `Počet plavcov: ${payload.club_size}` : "",
        payload.start_date ? `Začiatok: ${payload.start_date}` : "",
        payload.usage_intent ? `Využitie: ${payload.usage_intent}` : "",
        payload.current_planning ? `Plánovanie: ${payload.current_planning}` : "",
        payload.needs ? `Potreby: ${payload.needs}` : "",
        payload.ai_attitude ? `AI postoj: ${payload.ai_attitude}` : "",
      ]
        .filter(Boolean)
        .join("\n");
      const response = await fetch(FORMSPREE_ENDPOINT, {
        method: "POST",
        headers: { "Content-Type": "application/json", Accept: "application/json" },
        body: JSON.stringify({
          club: payload.club,
          contact: payload.contact,
          email: payload.email,
          notes,
          _subject: "CoachDesk Beta – nový záujem",
          _replyto: payload.email,
          recipient: "coachdeskswimming@gmail.com",
        }),
      });
      if (response.ok) {
        setStatus("Ďakujeme! Ozveme sa vám do 24–48 hodín.", false);
      } else {
        setStatus("Odoslanie zlyhalo. Skúste to prosím neskôr.", true);
        toggleButtons(false);
        isSubmitting = false;
        return;
      }
    } catch (err) {
      setStatus("Odoslanie zlyhalo. Skúste to prosím neskôr.", true);
      toggleButtons(false);
      isSubmitting = false;
      return;
    }
    isSubmitting = false;
  };

  submitButtons.forEach((btn) => {
    if (btn) {
      btn.addEventListener("click", handleSubmit);
    }
  });
  document.querySelectorAll(".auth-form").forEach((form) => {
    form.addEventListener("submit", (event) => {
      event.preventDefault();
      handleSubmit();
    });
  });
});
