const http = require("http");
const url = require("url");
const fs = require("fs");
const path = require("path");
const crypto = require("crypto");

const COOKIE_NAME = "coachdesk_session";
const DEMO_ADMIN_USERNAME = "demo_admin";
const DEMO_ADMIN_PASSWORD = "DemoAdmin2025!";

function createServer({ staticDir, seedDbPath, userDataDir }) {
  const dbPath = path.join(userDataDir, "demo_db.json");
  ensureDbSeed(dbPath, seedDbPath);
  const sessions = new Map();

  const server = http.createServer(async (req, res) => {
    const parsed = url.parse(req.url || "/", true);
    const rawPathname = decodeURIComponent(parsed.pathname || "/");
    const { pathname } = normalizePath(rawPathname);

    if (pathname.startsWith("/api/") || pathname.startsWith("/admin/") || pathname === "/login" || pathname === "/me" || pathname === "/logout") {
      await handleApi(req, res, pathname, parsed.query, dbPath, sessions);
      return;
    }
    serveStatic(req, res, pathname, staticDir);
  });

  return { server, sessions };
}

function normalizePath(pathname) {
  const prefix = "/demo";
  if (pathname === prefix) return { pathname: "/", hadDemo: true };
  if (pathname.startsWith(prefix + "/")) return { pathname: pathname.slice(prefix.length), hadDemo: true };
  return { pathname, hadDemo: false };
}

function ensureDbSeed(dbPath, seedDbPath) {
  if (fs.existsSync(dbPath)) return;
  fs.mkdirSync(path.dirname(dbPath), { recursive: true });
  fs.copyFileSync(seedDbPath, dbPath);
}

function loadDb(dbPath) {
  return JSON.parse(fs.readFileSync(dbPath, "utf-8"));
}

function saveDb(dbPath, db) {
  fs.writeFileSync(dbPath, JSON.stringify(db, null, 2), "utf-8");
}

function parseCookies(req) {
  const header = req.headers.cookie || "";
  const out = {};
  header.split(";").forEach((part) => {
    const [k, v] = part.split("=").map((s) => s && s.trim());
    if (!k) return;
    out[k] = decodeURIComponent(v || "");
  });
  return out;
}

function jsonResponse(res, status, payload) {
  const body = JSON.stringify(payload ?? {});
  res.writeHead(status, {
    "Content-Type": "application/json",
    "Content-Length": Buffer.byteLength(body),
  });
  res.end(body);
}

function textResponse(res, status, payload, contentType) {
  const body = payload || "";
  res.writeHead(status, {
    "Content-Type": contentType || "text/plain",
    "Content-Length": Buffer.byteLength(body),
  });
  res.end(body);
}

function setCookie(res, token) {
  res.setHeader("Set-Cookie", `${COOKIE_NAME}=${token}; HttpOnly; Path=/; SameSite=Lax`);
}

function clearCookie(res) {
  res.setHeader("Set-Cookie", `${COOKIE_NAME}=; HttpOnly; Path=/; Max-Age=0; SameSite=Lax`);
}

async function readJsonBody(req) {
  const chunks = [];
  for await (const chunk of req) {
    chunks.push(chunk);
  }
  if (!chunks.length) return {};
  try {
    return JSON.parse(Buffer.concat(chunks).toString("utf-8"));
  } catch (err) {
    return {};
  }
}

function requireAuth(req, res, sessions) {
  const cookies = parseCookies(req);
  const token = cookies[COOKIE_NAME];
  const session = token ? sessions.get(token) : null;
  if (!session) {
    jsonResponse(res, 401, { detail: "Unauthorized" });
    return null;
  }
  return session;
}

async function handleApi(req, res, pathname, query, dbPath, sessions) {
  const method = (req.method || "GET").toUpperCase();

  if ((pathname === "/api/login" || pathname === "/login") && method === "POST") {
    const payload = await readJsonBody(req);
    const username = String(payload.username || "");
    const password = String(payload.password || "");
    if (username !== DEMO_ADMIN_USERNAME || (password !== DEMO_ADMIN_PASSWORD && password !== "demo")) {
      jsonResponse(res, 401, { detail: "Invalid credentials" });
      return;
    }
    const db = loadDb(dbPath);
    const clubId = Object.keys(db.clubs || {})[0] || "club_default";
    const token = crypto.randomBytes(16).toString("hex");
    sessions.set(token, {
      username: DEMO_ADMIN_USERNAME,
      user_id: DEMO_ADMIN_USERNAME,
      role: "ADMIN",
      club_id: clubId,
    });
    setCookie(res, token);
    jsonResponse(res, 200, {
      role: "ADMIN",
      club_id: clubId,
      user_id: DEMO_ADMIN_USERNAME,
      token: token,
      club_tier: { tier: "basic" },
      access_token: token,
      expires_in: 3600,
    });
    return;
  }

  if ((pathname === "/api/logout" || pathname === "/logout") && method === "POST") {
    clearCookie(res);
    jsonResponse(res, 200, { ok: true });
    return;
  }

  if (matchPath(pathname, ["/register/club", "/api/register/club"]) && method === "POST") {
    jsonResponse(res, 403, { detail: "Registrácia je v demo režime vypnutá." });
    return;
  }

  const session = requireAuth(req, res, sessions);
  if (!session) return;

  if ((pathname === "/api/me" || pathname === "/me") && method === "GET") {
    jsonResponse(res, 200, {
      user_id: session.user_id,
      username: session.username,
      role: session.role,
      club_id: session.club_id,
      club_tier: { tier: "basic" },
    });
    return;
  }

  if ((pathname === "/api/club" || pathname === "/club") && method === "GET") {
    const club = getClub(loadDb(dbPath), session.club_id);
    jsonResponse(res, 200, club);
    return;
  }

  if (matchPath(pathname, ["/api/club/coaches", "/admin/coaches", "/club/coaches"]) && method === "GET") {
    const db = loadDb(dbPath);
    jsonResponse(res, 200, listCoaches(db, session.club_id));
    return;
  }

  if (pathname === "/api/club/coaches" && method === "POST") {
    const payload = await readJsonBody(req);
    const db = loadDb(dbPath);
    const result = createCoach(db, session.club_id, payload);
    saveDb(dbPath, db);
    jsonResponse(res, 200, result);
    return;
  }

  if (matchPath(pathname, ["/api/club/athletes", "/admin/athletes", "/club/athletes"]) && method === "GET") {
    const db = loadDb(dbPath);
    jsonResponse(res, 200, listAthletes(db, session.club_id));
    return;
  }

  if (pathname === "/api/club/athletes" && method === "POST") {
    const payload = await readJsonBody(req);
    const db = loadDb(dbPath);
    const result = createAthlete(db, session.club_id, payload);
    saveDb(dbPath, db);
    jsonResponse(res, 200, result);
    return;
  }

  if (pathname === "/api/club/access" && method === "GET") {
    const db = loadDb(dbPath);
    jsonResponse(res, 200, buildAccessAudit(db, session.club_id));
    return;
  }

  if (pathname === "/admin/assign-athlete" && method === "POST") {
    const payload = await readJsonBody(req);
    const db = loadDb(dbPath);
    const result = assignAthlete(db, session.club_id, payload);
    saveDb(dbPath, db);
    jsonResponse(res, 200, result);
    return;
  }

  if (pathname.startsWith("/admin/club/") && pathname.endsWith("/benchmark") && method === "GET") {
    jsonResponse(res, 200, buildBenchmark());
    return;
  }

  if (pathname.startsWith("/admin/export/") && method === "GET") {
    if (pathname.endsWith("/pdf")) {
      textResponse(res, 200, demoPdf(), "application/pdf");
    } else {
      textResponse(res, 200, demoCsv(pathname), "text/csv");
    }
    return;
  }

  jsonResponse(res, 404, { detail: "Not found" });
}

function matchPath(pathname, list) {
  return list.includes(pathname);
}

function getClub(db, clubId) {
  const clubs = db.clubs || {};
  const club = clubs[clubId] || clubs[Object.keys(clubs)[0]] || { id: clubId, name: "Demo Club" };
  return { ...club, id: club.id || clubId };
}

function listCoaches(db, clubId) {
  const coaches = Object.values(db.coaches || {}).filter((c) => c.club_id === clubId);
  const users = db.users || {};
  return coaches.map((coach) => {
    const user = users[coach.user_id] || {};
    return {
      id: coach.id,
      first_name: coach.first_name,
      last_name: coach.last_name,
      email: coach.email || "",
      phone: coach.phone || "",
      club_id: coach.club_id,
      username: user.username,
      is_active: user.is_active !== false,
    };
  });
}

function listAthletes(db, clubId) {
  const athletes = Object.values(db.athletes || {}).filter((a) => a.club_id === clubId);
  const users = db.users || {};
  return athletes.map((athlete) => {
    const user = users[athlete.user_id] || {};
    return {
      id: athlete.id,
      first_name: athlete.first_name,
      last_name: athlete.last_name,
      birth_date: athlete.birth_date,
      gender: athlete.gender,
      category: athlete.category,
      dominant_style: athlete.dominant_style,
      club_id: athlete.club_id,
      username: user.username,
      is_active: user.is_active !== false,
    };
  });
}

function buildAccessAudit(db, clubId) {
  const club = getClub(db, clubId);
  const users = Object.values(db.users || {}).filter((u) => u.club_id === clubId);
  const coachesByUser = {};
  for (const c of Object.values(db.coaches || {})) coachesByUser[c.user_id] = c;
  const athletesByUser = {};
  for (const a of Object.values(db.athletes || {})) athletesByUser[a.user_id] = a;
  return users.map((user) => {
    const coach = coachesByUser[user.id];
    const athlete = athletesByUser[user.id];
    const name = coach ? `${coach.first_name} ${coach.last_name}` : athlete ? `${athlete.first_name} ${athlete.last_name}` : user.username;
    const role = String(user.role || "").toUpperCase();
    const access = role === "ATHLETE" ? "App" : "Web + app";
    return {
      name,
      username: user.username,
      role,
      club: club.name || "Demo Club",
      access,
      last_login: user.last_login || "",
    };
  });
}

function assignAthlete(db, clubId, payload) {
  const coachId = payload.coach_id;
  const athleteId = payload.athlete_id;
  if (!coachId || !athleteId) return { ok: false };
  db.coach_athletes = db.coach_athletes || [];
  const exists = db.coach_athletes.some((rel) => rel.coach_id === coachId && rel.athlete_id === athleteId);
  if (!exists) {
    db.coach_athletes.push({
      coach_id: coachId,
      athlete_id: athleteId,
      assigned_at: new Date().toISOString(),
    });
  }
  return { ok: true };
}

function createCoach(db, clubId, payload) {
  const firstName = payload.first_name || "Coach";
  const lastName = payload.last_name || "Demo";
  const userId = crypto.randomUUID();
  const coachId = crypto.randomUUID();
  const username = `coach_${crypto.randomBytes(3).toString("hex")}`;
  const password = crypto.randomBytes(6).toString("base64url");
  db.users = db.users || {};
  db.users[userId] = {
    id: userId,
    username,
    password_hash_or_password: password,
    role: "COACH",
    club_id: clubId,
    is_active: true,
    created_at: new Date().toISOString(),
    last_login: "",
  };
  db.coaches = db.coaches || {};
  db.coaches[coachId] = {
    id: coachId,
    user_id: userId,
    club_id: clubId,
    first_name: firstName,
    last_name: lastName,
    email: payload.email || "",
    phone: payload.phone || "",
    created_at: new Date().toISOString(),
  };
  return {
    coach_id: coachId,
    credentials: { username, password },
  };
}

function createAthlete(db, clubId, payload) {
  const firstName = payload.first_name || "Athlete";
  const lastName = payload.last_name || "Demo";
  const userId = crypto.randomUUID();
  const athleteId = crypto.randomUUID();
  const username = `athlete_${crypto.randomBytes(3).toString("hex")}`;
  const password = crypto.randomBytes(6).toString("base64url");
  db.users = db.users || {};
  db.users[userId] = {
    id: userId,
    username,
    password_hash_or_password: password,
    role: "ATHLETE",
    club_id: clubId,
    is_active: true,
    created_at: new Date().toISOString(),
    last_login: "",
  };
  db.athletes = db.athletes || {};
  db.athletes[athleteId] = {
    id: athleteId,
    club_id: clubId,
    first_name: firstName,
    last_name: lastName,
    birth_date: payload.birth_date || "",
    gender: payload.gender || "",
    category: payload.category || "",
    dominant_style: payload.dominant_style || "",
    swimrankings_profile_url: payload.swimrankings_profile_url || "",
    user_id: userId,
    created_at: new Date().toISOString(),
  };
  return {
    athlete_id: athleteId,
    credentials: { username, password },
  };
}

function buildBenchmark() {
  const weeks = {};
  for (let i = 0; i < 8; i++) {
    weeks[`w${i + 1}`] = 18000 + i * 1200;
  }
  return {
    metrics: {
      avg_weekly_load: "24 500 m",
      injury_risk_proxy: "Low",
      performance_proxy: "Stable",
    },
    weeks,
  };
}

function demoCsv(pathname) {
  return `type,scope,path\nexport,demo,${pathname}\n`;
}

function demoPdf() {
  return [
    "%PDF-1.4",
    "1 0 obj << /Type /Catalog /Pages 2 0 R >> endobj",
    "2 0 obj << /Type /Pages /Kids [3 0 R] /Count 1 >> endobj",
    "3 0 obj << /Type /Page /Parent 2 0 R /MediaBox [0 0 300 144] /Contents 4 0 R /Resources << /Font << /F1 5 0 R >> >> >> endobj",
    "4 0 obj << /Length 44 >> stream",
    "BT /F1 12 Tf 20 100 Td (CoachDesk Demo Export) Tj ET",
    "endstream endobj",
    "5 0 obj << /Type /Font /Subtype /Type1 /BaseFont /Helvetica >> endobj",
    "xref",
    "0 6",
    "0000000000 65535 f ",
    "0000000010 00000 n ",
    "0000000060 00000 n ",
    "0000000117 00000 n ",
    "0000000221 00000 n ",
    "0000000320 00000 n ",
    "trailer << /Root 1 0 R /Size 6 >>",
    "startxref",
    "390",
    "%%EOF",
  ].join("\n");
}

function serveStatic(req, res, pathname, staticDir) {
  let safePath = pathname;
  if (safePath === "/") safePath = "/login.html";
  const filePath = path.join(staticDir, path.normalize(safePath));
  if (!filePath.startsWith(staticDir)) {
    textResponse(res, 403, "Forbidden");
    return;
  }
  fs.readFile(filePath, (err, data) => {
    if (err) {
      textResponse(res, 404, "Not found");
      return;
    }
    const ext = path.extname(filePath).toLowerCase();
    const contentType = {
      ".html": "text/html",
      ".css": "text/css",
      ".js": "application/javascript",
      ".json": "application/json",
      ".png": "image/png",
      ".jpg": "image/jpeg",
      ".jpeg": "image/jpeg",
      ".svg": "image/svg+xml",
      ".ico": "image/x-icon",
      ".woff": "font/woff",
      ".woff2": "font/woff2",
    }[ext] || "application/octet-stream";
    res.writeHead(200, { "Content-Type": contentType });
    res.end(data);
  });
}

module.exports = { createServer };
